<?php
/**
 * Homeschool & Chore Management System
 * Kid Dashboard (Main Page)
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/functions.php';

// Get all active children for navigation
$children = get_active_children($db);

// Determine current view
$current_child_id = isset($_GET['child']) ? (int)$_GET['child'] : null;
$current_child = null;
$show_home = ($current_child_id === null);

if ($current_child_id) {
    $current_child = get_child($db, $current_child_id);
    if (!$current_child || $current_child['is_archived']) {
        $current_child = null;
        $show_home = true;
    }
}

// Check if PIN is required for this child
$pin_required = false;
$pin_verified = false;
if ($current_child && !empty($current_child['pin'])) {
    $session_key = 'child_pin_verified_' . $current_child_id;
    $pin_verified = isset($_SESSION[$session_key]) && $_SESSION[$session_key] === true;
    $pin_required = !$pin_verified;
}

// Handle PIN submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['verify_pin'])) {
    $submitted_pin = $_POST['pin'] ?? '';
    if ($current_child && $submitted_pin === $current_child['pin']) {
        $_SESSION['child_pin_verified_' . $current_child_id] = true;
        $pin_required = false;
        $pin_verified = true;
    } else {
        $pin_error = 'Incorrect PIN';
    }
}

// Get today's date
$today = date('Y-m-d');

// Get tasks if viewing a child (and PIN verified if required)
$tasks = [];
$stats = ['total' => 0, 'completed' => 0];
if ($current_child && !$pin_required) {
    $tasks = get_child_tasks_for_date($db, $current_child_id, $today);
    $stats = get_child_task_stats($db, $current_child_id, $today);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no, viewport-fit=cover">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="default">
    <meta name="apple-mobile-web-app-title" content="<?php echo h(APP_NAME); ?>">
    <link rel="apple-touch-icon" href="<?php echo asset_url('img/app-icon.png'); ?>">
    <link rel="manifest" href="<?php echo APP_URL; ?>/manifest.json">
    <title><?php echo h(APP_NAME); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }
        
        html, body {
            height: 100%;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
            background: #f8fafc;
            overflow-x: hidden;
        }
        
        /* ========== TOP NAVIGATION ========== */
        .top-nav {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 12px 16px;
            background: #fff;
            border-bottom: 1px solid #e2e8f0;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .nav-home {
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            border-radius: 12px;
            transition: background 0.2s;
        }
        
        .nav-home:hover, .nav-home:active {
            background: #f1f5f9;
        }
        
        .nav-home.active {
            background: #e0e7ff;
        }
        
        .nav-home svg {
            width: 36px;
            height: 36px;
            fill: #475569;
        }
        
        .nav-home.active svg {
            fill: #4f46e5;
        }
        
        .nav-children {
            display: flex;
            gap: 12px;
            flex: 1;
            justify-content: center;
            padding: 0 16px;
            overflow-x: auto;
        }
        
        .nav-child {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            border: 3px solid #e2e8f0;
            cursor: pointer;
            overflow: hidden;
            flex-shrink: 0;
            transition: transform 0.2s, border-color 0.2s;
            background: #f1f5f9;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .nav-child:hover {
            transform: scale(1.05);
        }
        
        .nav-child.active {
            border-color: var(--child-color, #4f46e5);
            box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.2);
        }
        
        .nav-child img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .nav-child .placeholder-avatar {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #cbd5e1;
        }
        
        .nav-child .placeholder-avatar svg {
            width: 32px;
            height: 32px;
            fill: #64748b;
        }
        
        .nav-menu {
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            border-radius: 12px;
            transition: background 0.2s;
        }
        
        .nav-menu:hover, .nav-menu:active {
            background: #f1f5f9;
        }
        
        .nav-menu svg {
            width: 32px;
            height: 32px;
            stroke: #475569;
        }
        
        /* ========== MAIN CONTENT ========== */
        .main-content {
            padding: 20px;
            max-width: 800px;
            margin: 0 auto;
            min-height: calc(100vh - 85px);
        }
        
        /* ========== HOME VIEW ========== */
        .home-view {
            text-align: center;
            padding-top: 60px;
        }
        
        .home-view h1 {
            font-size: 32px;
            color: #1e293b;
            margin-bottom: 16px;
        }
        
        .home-view p {
            font-size: 20px;
            color: #64748b;
        }
        
        .home-date {
            font-size: 24px;
            color: #475569;
            margin-top: 24px;
        }
        
        /* ========== CHILD VIEW ========== */
        .child-header {
            text-align: center;
            margin-bottom: 24px;
        }
        
        .child-header h1 {
            font-size: 28px;
            color: #1e293b;
            margin-bottom: 8px;
        }
        
        .progress-bar {
            width: 100%;
            max-width: 400px;
            height: 12px;
            background: #e2e8f0;
            border-radius: 6px;
            margin: 16px auto;
            overflow: hidden;
        }
        
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #22c55e, #16a34a);
            border-radius: 6px;
            transition: width 0.5s ease;
        }
        
        .progress-text {
            font-size: 18px;
            color: #64748b;
        }
        
        /* ========== TASK LIST ========== */
        .task-list {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }
        
        .task-card {
            background: #fff;
            border-radius: 16px;
            padding: 20px;
            display: flex;
            align-items: center;
            gap: 16px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
            cursor: pointer;
            transition: transform 0.2s, box-shadow 0.2s;
            border-left: 4px solid var(--task-color, #6366f1);
        }
        
        .task-card:active {
            transform: scale(0.98);
        }
        
        .task-card.completed {
            opacity: 0.6;
            background: #f8fafc;
        }
        
        .task-card.completed .task-title {
            text-decoration: line-through;
            color: #94a3b8;
        }
        
        .task-checkbox {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            border: 3px solid #cbd5e1;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            transition: all 0.2s;
        }
        
        .task-card.completed .task-checkbox {
            background: #22c55e;
            border-color: #22c55e;
        }
        
        .task-checkbox svg {
            width: 24px;
            height: 24px;
            stroke: #fff;
            opacity: 0;
            transition: opacity 0.2s;
        }
        
        .task-card.completed .task-checkbox svg {
            opacity: 1;
        }
        
        .task-info {
            flex: 1;
        }
        
        .task-title {
            font-size: 20px;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 4px;
        }
        
        .task-emoji {
            margin-right: 8px;
        }
        
        .task-description {
            font-size: 16px;
            color: #64748b;
        }
        
        .task-type {
            font-size: 12px;
            padding: 4px 10px;
            border-radius: 12px;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .task-type.homeschool {
            background: #dbeafe;
            color: #1d4ed8;
        }
        
        .task-type.chore {
            background: #fef3c7;
            color: #b45309;
        }
        
        .no-tasks {
            text-align: center;
            padding: 60px 20px;
            color: #64748b;
            font-size: 20px;
        }
        
        /* ========== PIN ENTRY ========== */
        .pin-entry {
            text-align: center;
            padding-top: 60px;
        }
        
        .pin-entry h2 {
            font-size: 24px;
            color: #1e293b;
            margin-bottom: 24px;
        }
        
        .pin-form {
            max-width: 300px;
            margin: 0 auto;
        }
        
        .pin-input {
            width: 100%;
            font-size: 32px;
            text-align: center;
            padding: 16px;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            letter-spacing: 8px;
            margin-bottom: 16px;
        }
        
        .pin-input:focus {
            outline: none;
            border-color: #6366f1;
        }
        
        .pin-error {
            color: #dc2626;
            margin-bottom: 16px;
        }
        
        .pin-submit {
            width: 100%;
            padding: 16px;
            font-size: 18px;
            background: #6366f1;
            color: #fff;
            border: none;
            border-radius: 12px;
            cursor: pointer;
        }
        
        /* ========== SLIDE-OUT MENU ========== */
        .menu-overlay {
            position: fixed;
            inset: 0;
            background: rgba(0,0,0,0.5);
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s;
            z-index: 200;
        }
        
        .menu-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .slide-menu {
            position: fixed;
            top: 0;
            right: -300px;
            width: 300px;
            height: 100%;
            background: #fff;
            box-shadow: -4px 0 20px rgba(0,0,0,0.1);
            transition: right 0.3s;
            z-index: 201;
            padding: 24px;
        }
        
        .slide-menu.active {
            right: 0;
        }
        
        .menu-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 32px;
        }
        
        .menu-header h2 {
            font-size: 24px;
            color: #1e293b;
        }
        
        .menu-close {
            width: 44px;
            height: 44px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            border-radius: 8px;
        }
        
        .menu-close:hover {
            background: #f1f5f9;
        }
        
        .menu-close svg {
            width: 24px;
            height: 24px;
            stroke: #64748b;
        }
        
        .menu-links {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        
        .menu-link {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 16px;
            border-radius: 12px;
            text-decoration: none;
            color: #1e293b;
            font-size: 18px;
            transition: background 0.2s;
        }
        
        .menu-link:hover {
            background: #f1f5f9;
        }
        
        .menu-link svg {
            width: 24px;
            height: 24px;
            stroke: #64748b;
        }
        
        /* ========== CELEBRATION ========== */
        .celebration {
            position: fixed;
            inset: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            background: rgba(0,0,0,0.5);
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s;
            z-index: 300;
        }
        
        .celebration.active {
            opacity: 1;
            visibility: visible;
        }
        
        .celebration-content {
            background: #fff;
            padding: 48px;
            border-radius: 24px;
            text-align: center;
            transform: scale(0.8);
            transition: transform 0.3s;
        }
        
        .celebration.active .celebration-content {
            transform: scale(1);
        }
        
        .celebration-emoji {
            font-size: 72px;
            margin-bottom: 16px;
        }
        
        .celebration h2 {
            font-size: 28px;
            color: #1e293b;
        }
    </style>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <!-- Home Button -->
        <a href="<?php echo APP_URL; ?>/" class="nav-home <?php echo $show_home ? 'active' : ''; ?>">
            <svg viewBox="0 0 24 24">
                <path d="M10 20v-6h4v6h5v-8h3L12 3 2 12h3v8z"/>
            </svg>
        </a>
        
        <!-- Child Avatars -->
        <div class="nav-children">
            <?php foreach ($children as $child): ?>
                <a href="<?php echo APP_URL; ?>/?child=<?php echo $child['id']; ?>" 
                   class="nav-child <?php echo $current_child_id == $child['id'] ? 'active' : ''; ?>"
                   style="--child-color: <?php echo h($child['color']); ?>">
                    <?php if (!empty($child['photo'])): ?>
                        <img src="<?php echo upload_url($child['photo']); ?>" alt="<?php echo h($child['name']); ?>">
                    <?php else: ?>
                        <div class="placeholder-avatar">
                            <svg viewBox="0 0 24 24">
                                <path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"/>
                            </svg>
                        </div>
                    <?php endif; ?>
                </a>
            <?php endforeach; ?>
        </div>
        
        <!-- Menu Button -->
        <div class="nav-menu" onclick="toggleMenu()">
            <svg viewBox="0 0 24 24" fill="none" stroke-width="2" stroke-linecap="round">
                <line x1="3" y1="6" x2="21" y2="6"/>
                <line x1="3" y1="12" x2="21" y2="12"/>
                <line x1="3" y1="18" x2="21" y2="18"/>
            </svg>
        </div>
    </nav>
    
    <!-- Main Content -->
    <main class="main-content">
        <?php if ($show_home): ?>
            <!-- Home View -->
            <div class="home-view">
                <h1>👋 Welcome!</h1>
                <p>Tap your picture above to see your tasks</p>
                <p class="home-date"><?php echo date('l, F j'); ?></p>
            </div>
            
        <?php elseif ($pin_required): ?>
            <!-- PIN Entry -->
            <div class="pin-entry">
                <h2>Enter PIN for <?php echo h($current_child['name']); ?></h2>
                <form method="POST" class="pin-form">
                    <?php if (isset($pin_error)): ?>
                        <p class="pin-error"><?php echo h($pin_error); ?></p>
                    <?php endif; ?>
                    <input type="password" name="pin" class="pin-input" maxlength="4" pattern="[0-9]{4}" inputmode="numeric" autofocus>
                    <button type="submit" name="verify_pin" class="pin-submit">Enter</button>
                </form>
            </div>
            
        <?php else: ?>
            <!-- Child Task View -->
            <div class="child-header" style="--child-color: <?php echo h($current_child['color']); ?>">
                <h1><?php echo h($current_child['name']); ?>'s Tasks</h1>
                
                <?php if ($stats['total'] > 0): ?>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php echo ($stats['completed'] / $stats['total']) * 100; ?>%"></div>
                    </div>
                    <p class="progress-text"><?php echo $stats['completed']; ?> of <?php echo $stats['total']; ?> done</p>
                <?php endif; ?>
            </div>
            
            <?php if (empty($tasks)): ?>
                <div class="no-tasks">
                    <p>🎉 No tasks for today!</p>
                </div>
            <?php else: ?>
                <div class="task-list">
                    <?php foreach ($tasks as $task): ?>
                        <div class="task-card <?php echo $task['is_completed'] ? 'completed' : ''; ?>" 
                             data-instance-id="<?php echo $task['id']; ?>"
                             style="--task-color: <?php echo $task['type'] === 'homeschool' ? '#3b82f6' : '#f59e0b'; ?>"
                             onclick="toggleTask(this)">
                            <div class="task-checkbox">
                                <svg viewBox="0 0 24 24" fill="none" stroke-width="3" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="20 6 9 17 4 12"/>
                                </svg>
                            </div>
                            <div class="task-info">
                                <div class="task-title">
                                    <?php if (!empty($task['emoji'])): ?>
                                        <span class="task-emoji"><?php echo h($task['emoji']); ?></span>
                                    <?php endif; ?>
                                    <?php echo h($task['title']); ?>
                                </div>
                                <?php if (!empty($task['description'])): ?>
                                    <div class="task-description"><?php echo h($task['description']); ?></div>
                                <?php endif; ?>
                            </div>
                            <span class="task-type <?php echo h($task['type']); ?>"><?php echo h($task['type']); ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </main>
    
    <!-- Slide-out Menu -->
    <div class="menu-overlay" onclick="toggleMenu()"></div>
    <div class="slide-menu">
        <div class="menu-header">
            <h2>Menu</h2>
            <div class="menu-close" onclick="toggleMenu()">
                <svg viewBox="0 0 24 24" fill="none" stroke-width="2" stroke-linecap="round">
                    <line x1="18" y1="6" x2="6" y2="18"/>
                    <line x1="6" y1="6" x2="18" y2="18"/>
                </svg>
            </div>
        </div>
        <div class="menu-links">
            <a href="<?php echo APP_URL; ?>/admin/" class="menu-link">
                <svg viewBox="0 0 24 24" fill="none" stroke-width="2" stroke-linecap="round">
                    <circle cx="12" cy="12" r="3"/>
                    <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/>
                </svg>
                Admin
            </a>
            <a href="#" class="menu-link" onclick="enterFullscreen(); return false;">
                <svg viewBox="0 0 24 24" fill="none" stroke-width="2" stroke-linecap="round">
                    <polyline points="15 3 21 3 21 9"/>
                    <polyline points="9 21 3 21 3 15"/>
                    <line x1="21" y1="3" x2="14" y2="10"/>
                    <line x1="3" y1="21" x2="10" y2="14"/>
                </svg>
                Fullscreen
            </a>
        </div>
    </div>
    
    <!-- Celebration Modal -->
    <div class="celebration" id="celebration" onclick="hideCelebration()">
        <div class="celebration-content">
            <div class="celebration-emoji">🎉</div>
            <h2>All Done! Great Job!</h2>
        </div>
    </div>
    
    <script>
        // Menu toggle
        function toggleMenu() {
            document.querySelector('.menu-overlay').classList.toggle('active');
            document.querySelector('.slide-menu').classList.toggle('active');
        }
        
        // Fullscreen
        function enterFullscreen() {
            const elem = document.documentElement;
            if (elem.requestFullscreen) {
                elem.requestFullscreen();
            } else if (elem.webkitRequestFullscreen) {
                elem.webkitRequestFullscreen();
            }
            toggleMenu();
        }
        
        // Toggle task completion
        async function toggleTask(card) {
            const instanceId = card.dataset.instanceId;
            const isCompleted = card.classList.contains('completed');
            const newState = !isCompleted;
            
            // Optimistic UI update
            card.classList.toggle('completed');
            updateProgress();
            
            try {
                const response = await fetch('<?php echo APP_URL; ?>/api/complete_task.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        instance_id: instanceId,
                        is_completed: newState ? 1 : 0
                    })
                });
                
                const data = await response.json();
                
                if (!data.success) {
                    // Revert on failure
                    card.classList.toggle('completed');
                    updateProgress();
                    alert('Failed to update task');
                } else if (newState) {
                    // Check if all tasks are done
                    checkAllComplete();
                }
            } catch (error) {
                // Revert on error
                card.classList.toggle('completed');
                updateProgress();
                console.error('Error:', error);
            }
        }
        
        // Update progress bar
        function updateProgress() {
            const tasks = document.querySelectorAll('.task-card');
            const completed = document.querySelectorAll('.task-card.completed');
            const total = tasks.length;
            const done = completed.length;
            
            const progressFill = document.querySelector('.progress-fill');
            const progressText = document.querySelector('.progress-text');
            
            if (progressFill && total > 0) {
                progressFill.style.width = (done / total * 100) + '%';
            }
            if (progressText) {
                progressText.textContent = done + ' of ' + total + ' done';
            }
        }
        
        // Check if all tasks are complete
        function checkAllComplete() {
            const tasks = document.querySelectorAll('.task-card');
            const completed = document.querySelectorAll('.task-card.completed');
            
            if (tasks.length > 0 && tasks.length === completed.length) {
                showCelebration();
            }
        }
        
        // Show celebration
        function showCelebration() {
            document.getElementById('celebration').classList.add('active');
            setTimeout(hideCelebration, 3000);
        }
        
        // Hide celebration
        function hideCelebration() {
            document.getElementById('celebration').classList.remove('active');
        }
        
        // Wake lock (keep screen on)
        async function requestWakeLock() {
            if ('wakeLock' in navigator) {
                try {
                    await navigator.wakeLock.request('screen');
                } catch (err) {
                    console.log('Wake Lock error:', err);
                }
            }
        }
        
        // Request wake lock on load
        requestWakeLock();
        
        // Re-request wake lock when page becomes visible
        document.addEventListener('visibilitychange', () => {
            if (document.visibilityState === 'visible') {
                requestWakeLock();
            }
        });
    </script>
</body>
</html>
