<?php
/**
 * Homeschool & Chore Management System
 * Core Functions
 */

// =====================================================
// CHILDREN FUNCTIONS
// =====================================================

/**
 * Get all active children (for navigation)
 */
function get_active_children($db) {
    $sql = "SELECT * FROM children WHERE is_archived = 0 ORDER BY sort_order ASC, name ASC";
    $result = mysqli_query($db, $sql);
    $children = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $children[] = $row;
    }
    return $children;
}

/**
 * Get all children (including archived, for admin)
 */
function get_all_children($db, $include_archived = false) {
    $where = $include_archived ? "1=1" : "is_archived = 0";
    $sql = "SELECT * FROM children WHERE $where ORDER BY is_archived ASC, sort_order ASC, name ASC";
    $result = mysqli_query($db, $sql);
    $children = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $children[] = $row;
    }
    return $children;
}

/**
 * Get single child by ID
 */
function get_child($db, $id) {
    $id = (int)$id;
    $sql = "SELECT * FROM children WHERE id = $id";
    $result = mysqli_query($db, $sql);
    return mysqli_fetch_assoc($result);
}

/**
 * Create a new child
 */
function create_child($db, $data) {
    $name = mysqli_real_escape_string($db, $data['name']);
    $photo = mysqli_real_escape_string($db, $data['photo'] ?? '');
    $color = mysqli_real_escape_string($db, $data['color'] ?? '#6366f1');
    $pin = !empty($data['pin']) ? mysqli_real_escape_string($db, $data['pin']) : 'NULL';
    $sort_order = (int)($data['sort_order'] ?? 0);
    
    $pin_sql = $pin === 'NULL' ? 'NULL' : "'$pin'";
    
    $sql = "INSERT INTO children (name, photo, color, pin, sort_order) 
            VALUES ('$name', '$photo', '$color', $pin_sql, $sort_order)";
    
    if (mysqli_query($db, $sql)) {
        return mysqli_insert_id($db);
    }
    return false;
}

/**
 * Update a child
 */
function update_child($db, $id, $data) {
    $id = (int)$id;
    $name = mysqli_real_escape_string($db, $data['name']);
    $color = mysqli_real_escape_string($db, $data['color'] ?? '#6366f1');
    $pin = !empty($data['pin']) ? "'" . mysqli_real_escape_string($db, $data['pin']) . "'" : 'NULL';
    $sort_order = (int)($data['sort_order'] ?? 0);
    
    $photo_sql = '';
    if (isset($data['photo'])) {
        $photo = mysqli_real_escape_string($db, $data['photo']);
        $photo_sql = ", photo = '$photo'";
    }
    
    $sql = "UPDATE children SET 
            name = '$name', 
            color = '$color', 
            pin = $pin, 
            sort_order = $sort_order
            $photo_sql
            WHERE id = $id";
    
    return mysqli_query($db, $sql);
}

/**
 * Archive a child (soft delete)
 */
function archive_child($db, $id) {
    $id = (int)$id;
    $sql = "UPDATE children SET is_archived = 1 WHERE id = $id";
    return mysqli_query($db, $sql);
}

/**
 * Restore an archived child
 */
function restore_child($db, $id) {
    $id = (int)$id;
    $sql = "UPDATE children SET is_archived = 0 WHERE id = $id";
    return mysqli_query($db, $sql);
}

/**
 * Verify child PIN
 */
function verify_child_pin($db, $child_id, $pin) {
    $child = get_child($db, $child_id);
    if (!$child || empty($child['pin'])) {
        return true; // No PIN required
    }
    return $child['pin'] === $pin;
}

// =====================================================
// TASKS FUNCTIONS
// =====================================================

/**
 * Get all tasks (for admin)
 */
function get_all_tasks($db, $filters = []) {
    $where = ["1=1"];
    
    if (isset($filters['is_active'])) {
        $where[] = "t.is_active = " . (int)$filters['is_active'];
    }
    if (!empty($filters['type'])) {
        $type = mysqli_real_escape_string($db, $filters['type']);
        $where[] = "t.type = '$type'";
    }
    if (isset($filters['child_id'])) {
        $child_id = (int)$filters['child_id'];
        $where[] = "(t.assigned_child_id = $child_id OR t.assigned_child_id IS NULL)";
    }
    
    $where_sql = implode(' AND ', $where);
    
    $sql = "SELECT t.*, c.name as child_name 
            FROM tasks t 
            LEFT JOIN children c ON t.assigned_child_id = c.id 
            WHERE $where_sql 
            ORDER BY t.sort_order ASC, t.title ASC";
    
    $result = mysqli_query($db, $sql);
    $tasks = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $tasks[] = $row;
    }
    return $tasks;
}

/**
 * Get single task by ID
 */
function get_task($db, $id) {
    $id = (int)$id;
    $sql = "SELECT * FROM tasks WHERE id = $id";
    $result = mysqli_query($db, $sql);
    return mysqli_fetch_assoc($result);
}

/**
 * Create a new task
 */
function create_task($db, $data) {
    $title = mysqli_real_escape_string($db, $data['title']);
    $emoji = mysqli_real_escape_string($db, $data['emoji'] ?? '');
    $description = mysqli_real_escape_string($db, $data['description'] ?? '');
    $type = mysqli_real_escape_string($db, $data['type'] ?? 'chore');
    $assigned_child_id = !empty($data['assigned_child_id']) ? (int)$data['assigned_child_id'] : 'NULL';
    $recurrence_type = mysqli_real_escape_string($db, $data['recurrence_type'] ?? 'none');
    $recurrence_days = mysqli_real_escape_string($db, $data['recurrence_days'] ?? '');
    $start_date = mysqli_real_escape_string($db, $data['start_date']);
    $end_date = !empty($data['end_date']) ? "'" . mysqli_real_escape_string($db, $data['end_date']) . "'" : 'NULL';
    $sort_order = (int)($data['sort_order'] ?? 0);
    
    $sql = "INSERT INTO tasks (title, emoji, description, type, assigned_child_id, recurrence_type, recurrence_days, start_date, end_date, sort_order) 
            VALUES ('$title', '$emoji', '$description', '$type', $assigned_child_id, '$recurrence_type', '$recurrence_days', '$start_date', $end_date, $sort_order)";
    
    if (mysqli_query($db, $sql)) {
        return mysqli_insert_id($db);
    }
    return false;
}

/**
 * Update a task
 */
function update_task($db, $id, $data) {
    $id = (int)$id;
    $title = mysqli_real_escape_string($db, $data['title']);
    $emoji = mysqli_real_escape_string($db, $data['emoji'] ?? '');
    $description = mysqli_real_escape_string($db, $data['description'] ?? '');
    $type = mysqli_real_escape_string($db, $data['type'] ?? 'chore');
    $assigned_child_id = !empty($data['assigned_child_id']) ? (int)$data['assigned_child_id'] : 'NULL';
    $recurrence_type = mysqli_real_escape_string($db, $data['recurrence_type'] ?? 'none');
    $recurrence_days = mysqli_real_escape_string($db, $data['recurrence_days'] ?? '');
    $start_date = mysqli_real_escape_string($db, $data['start_date']);
    $end_date = !empty($data['end_date']) ? "'" . mysqli_real_escape_string($db, $data['end_date']) . "'" : 'NULL';
    $sort_order = (int)($data['sort_order'] ?? 0);
    $is_active = isset($data['is_active']) ? (int)$data['is_active'] : 1;
    
    $sql = "UPDATE tasks SET 
            title = '$title',
            emoji = '$emoji',
            description = '$description',
            type = '$type',
            assigned_child_id = $assigned_child_id,
            recurrence_type = '$recurrence_type',
            recurrence_days = '$recurrence_days',
            start_date = '$start_date',
            end_date = $end_date,
            sort_order = $sort_order,
            is_active = $is_active
            WHERE id = $id";
    
    return mysqli_query($db, $sql);
}

/**
 * Delete a task (and its instances)
 */
function delete_task($db, $id) {
    $id = (int)$id;
    // Instances will be deleted by CASCADE
    $sql = "DELETE FROM tasks WHERE id = $id";
    return mysqli_query($db, $sql);
}

// =====================================================
// TASK INSTANCES FUNCTIONS
// =====================================================

/**
 * Get task instances for a child on a specific date
 */
function get_child_tasks_for_date($db, $child_id, $date) {
    $child_id = (int)$child_id;
    $date = mysqli_real_escape_string($db, $date);
    
    $sql = "SELECT ti.*, t.title, t.emoji, t.description, t.type 
            FROM task_instances ti 
            JOIN tasks t ON ti.task_id = t.id 
            WHERE ti.child_id = $child_id 
            AND ti.due_date = '$date' 
            ORDER BY t.sort_order ASC, t.title ASC";
    
    $result = mysqli_query($db, $sql);
    $tasks = [];
    while ($row = mysqli_fetch_assoc($result)) {
        $tasks[] = $row;
    }
    return $tasks;
}

/**
 * Get task completion stats for a child on a date
 */
function get_child_task_stats($db, $child_id, $date) {
    $child_id = (int)$child_id;
    $date = mysqli_real_escape_string($db, $date);
    
    $sql = "SELECT 
            COUNT(*) as total,
            SUM(is_completed) as completed
            FROM task_instances 
            WHERE child_id = $child_id AND due_date = '$date'";
    
    $result = mysqli_query($db, $sql);
    $row = mysqli_fetch_assoc($result);
    
    return [
        'total' => (int)$row['total'],
        'completed' => (int)$row['completed']
    ];
}

/**
 * Toggle task instance completion
 */
function toggle_task_completion($db, $instance_id, $completed) {
    $instance_id = (int)$instance_id;
    $is_completed = $completed ? 1 : 0;
    $completed_at = $completed ? "NOW()" : "NULL";
    
    $sql = "UPDATE task_instances SET 
            is_completed = $is_completed, 
            completed_at = $completed_at 
            WHERE id = $instance_id";
    
    return mysqli_query($db, $sql);
}

/**
 * Generate task instances for a specific date
 * Called by cron job
 */
function generate_task_instances_for_date($db, $date) {
    $date_obj = new DateTime($date);
    $day_of_week = $date_obj->format('w'); // 0=Sunday, 6=Saturday
    $date_sql = mysqli_real_escape_string($db, $date);
    
    // Get active children
    $children = get_active_children($db);
    if (empty($children)) {
        return 0;
    }
    
    // Get active tasks that should run on this date
    $sql = "SELECT * FROM tasks 
            WHERE is_active = 1 
            AND start_date <= '$date_sql' 
            AND (end_date IS NULL OR end_date >= '$date_sql')";
    
    $result = mysqli_query($db, $sql);
    $count = 0;
    
    while ($task = mysqli_fetch_assoc($result)) {
        // Check if task should run today based on recurrence
        $should_run = false;
        
        switch ($task['recurrence_type']) {
            case 'daily':
                $should_run = true;
                break;
            case 'weekdays':
                // Monday-Friday (1-5)
                $should_run = ($day_of_week >= 1 && $day_of_week <= 5);
                break;
            case 'weekly':
                // Check if specific days match
                if (!empty($task['recurrence_days'])) {
                    $days = explode(',', $task['recurrence_days']);
                    $should_run = in_array($day_of_week, $days);
                }
                break;
            case 'none':
                // One-time task - only run on start_date
                $should_run = ($task['start_date'] === $date);
                break;
        }
        
        if (!$should_run) {
            continue;
        }
        
        // Determine which children get this task
        if ($task['assigned_child_id']) {
            $target_children = [['id' => $task['assigned_child_id']]];
        } else {
            $target_children = $children; // All children
        }
        
        // Create instance for each target child
        foreach ($target_children as $child) {
            // Check if instance already exists (prevent duplicates)
            $check_sql = "SELECT id FROM task_instances 
                          WHERE task_id = {$task['id']} 
                          AND child_id = {$child['id']} 
                          AND due_date = '$date_sql'";
            $check_result = mysqli_query($db, $check_sql);
            
            if (mysqli_num_rows($check_result) === 0) {
                $insert_sql = "INSERT INTO task_instances (task_id, child_id, due_date) 
                               VALUES ({$task['id']}, {$child['id']}, '$date_sql')";
                if (mysqli_query($db, $insert_sql)) {
                    $count++;
                }
            }
        }
    }
    
    return $count;
}

// =====================================================
// FILE UPLOAD FUNCTIONS
// =====================================================

/**
 * Handle photo upload
 */
function handle_photo_upload($file, $prefix = 'photo') {
    // Check for errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'error' => 'Upload failed'];
    }
    
    // Check file size
    if ($file['size'] > MAX_UPLOAD_SIZE) {
        return ['success' => false, 'error' => 'File too large (max 5MB)'];
    }
    
    // Check file type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    if (!in_array($mime_type, ALLOWED_IMAGE_TYPES)) {
        return ['success' => false, 'error' => 'Invalid file type'];
    }
    
    // Generate unique filename
    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = $prefix . '_' . time() . '_' . bin2hex(random_bytes(8)) . '.' . $ext;
    
    // Create upload directory if needed
    if (!is_dir(UPLOAD_PATH)) {
        mkdir(UPLOAD_PATH, 0755, true);
    }
    
    // Move file
    $destination = UPLOAD_PATH . $filename;
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        return ['success' => true, 'filename' => $filename];
    }
    
    return ['success' => false, 'error' => 'Failed to save file'];
}

/**
 * Delete uploaded file
 */
function delete_uploaded_file($filename) {
    if (empty($filename)) {
        return true;
    }
    $path = UPLOAD_PATH . $filename;
    if (file_exists($path)) {
        return unlink($path);
    }
    return true;
}

// =====================================================
// SETTINGS FUNCTIONS
// =====================================================

/**
 * Get a setting value
 */
function get_setting($db, $key, $default = null) {
    $key = mysqli_real_escape_string($db, $key);
    $sql = "SELECT setting_value FROM settings WHERE setting_key = '$key'";
    $result = mysqli_query($db, $sql);
    if ($row = mysqli_fetch_assoc($result)) {
        return $row['setting_value'];
    }
    return $default;
}

/**
 * Set a setting value
 */
function set_setting($db, $key, $value) {
    $key = mysqli_real_escape_string($db, $key);
    $value = mysqli_real_escape_string($db, $value);
    $sql = "INSERT INTO settings (setting_key, setting_value) 
            VALUES ('$key', '$value') 
            ON DUPLICATE KEY UPDATE setting_value = '$value'";
    return mysqli_query($db, $sql);
}

// =====================================================
// USER AUTHENTICATION FUNCTIONS
// =====================================================

/**
 * Attempt admin login
 */
function attempt_admin_login($db, $username, $password) {
    $username = mysqli_real_escape_string($db, $username);
    $sql = "SELECT id, password_hash FROM users WHERE username = '$username'";
    $result = mysqli_query($db, $sql);
    
    if ($row = mysqli_fetch_assoc($result)) {
        if (password_verify($password, $row['password_hash'])) {
            // Regenerate session ID for security
            session_regenerate_id(true);
            $_SESSION['admin_id'] = $row['id'];
            $_SESSION['admin_username'] = $username;
            return true;
        }
    }
    return false;
}

/**
 * Logout admin
 */
function admin_logout() {
    unset($_SESSION['admin_id']);
    unset($_SESSION['admin_username']);
    session_regenerate_id(true);
}
