<?php
/**
 * Admin - Tasks List
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

$page_title = 'Tasks';

// Handle delete action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash('error', 'Invalid request');
    } else {
        $task_id = (int)$_POST['task_id'];
        delete_task($db, $task_id);
        set_flash('success', 'Task deleted successfully');
    }
    redirect('admin/tasks.php');
}

// Get filters
$filters = [];
if (isset($_GET['type']) && in_array($_GET['type'], ['homeschool', 'chore'])) {
    $filters['type'] = $_GET['type'];
}
if (isset($_GET['child_id']) && $_GET['child_id'] !== '') {
    $filters['child_id'] = (int)$_GET['child_id'];
}
if (!isset($_GET['show_inactive'])) {
    $filters['is_active'] = 1;
}

// Get tasks and children
$tasks = get_all_tasks($db, $filters);
$children = get_active_children($db);

require_once __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1>Tasks</h1>
    <a href="<?php echo APP_URL; ?>/admin/task_edit.php" class="btn btn-primary">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <line x1="12" y1="5" x2="12" y2="19"/>
            <line x1="5" y1="12" x2="19" y2="12"/>
        </svg>
        Add Task
    </a>
</div>

<!-- Filters -->
<div class="card" style="padding: 16px;">
    <form method="GET" style="display: flex; gap: 16px; flex-wrap: wrap; align-items: end;">
        <div style="flex: 1; min-width: 150px;">
            <label style="display: block; font-size: 12px; color: #64748b; margin-bottom: 4px;">Type</label>
            <select name="type" style="width: 100%; padding: 8px; border: 1px solid #e2e8f0; border-radius: 6px;">
                <option value="">All Types</option>
                <option value="homeschool" <?php echo ($_GET['type'] ?? '') === 'homeschool' ? 'selected' : ''; ?>>Homeschool</option>
                <option value="chore" <?php echo ($_GET['type'] ?? '') === 'chore' ? 'selected' : ''; ?>>Chore</option>
            </select>
        </div>
        
        <div style="flex: 1; min-width: 150px;">
            <label style="display: block; font-size: 12px; color: #64748b; margin-bottom: 4px;">Child</label>
            <select name="child_id" style="width: 100%; padding: 8px; border: 1px solid #e2e8f0; border-radius: 6px;">
                <option value="">All Children</option>
                <?php foreach ($children as $c): ?>
                    <option value="<?php echo $c['id']; ?>" <?php echo ($_GET['child_id'] ?? '') == $c['id'] ? 'selected' : ''; ?>>
                        <?php echo h($c['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div style="display: flex; align-items: center; gap: 8px;">
            <input type="checkbox" id="show_inactive" name="show_inactive" value="1" <?php echo isset($_GET['show_inactive']) ? 'checked' : ''; ?>>
            <label for="show_inactive" style="font-size: 14px; color: #64748b;">Show inactive</label>
        </div>
        
        <button type="submit" class="btn btn-secondary btn-sm">Filter</button>
        <a href="<?php echo APP_URL; ?>/admin/tasks.php" class="btn btn-secondary btn-sm">Reset</a>
    </form>
</div>

<div class="card">
    <div class="card-header">
        <h2>All Tasks</h2>
        <span style="color: #64748b; font-size: 14px;"><?php echo count($tasks); ?> task(s)</span>
    </div>
    
    <?php if (empty($tasks)): ?>
        <div class="empty-state">
            <svg viewBox="0 0 24 24" fill="none" stroke-width="2" stroke-linecap="round">
                <path d="M9 11l3 3L22 4"/>
                <path d="M21 12v7a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11"/>
            </svg>
            <h3>No Tasks Found</h3>
            <p>Create your first task to get started</p>
        </div>
    <?php else: ?>
        <div class="table-wrapper">
            <table>
                <thead>
                    <tr>
                        <th>Task</th>
                        <th>Type</th>
                        <th>Assigned To</th>
                        <th>Recurrence</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($tasks as $task): ?>
                        <tr style="<?php echo !$task['is_active'] ? 'opacity: 0.5;' : ''; ?>">
                            <td>
                                <div style="display: flex; align-items: center; gap: 8px;">
                                    <?php if (!empty($task['emoji'])): ?>
                                        <span style="font-size: 20px;"><?php echo h($task['emoji']); ?></span>
                                    <?php endif; ?>
                                    <div>
                                        <strong><?php echo h($task['title']); ?></strong>
                                        <?php if (!empty($task['description'])): ?>
                                            <div style="font-size: 13px; color: #64748b; margin-top: 2px;">
                                                <?php echo h(substr($task['description'], 0, 50)); ?><?php echo strlen($task['description']) > 50 ? '...' : ''; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </td>
                            <td>
                                <span class="badge badge-<?php echo h($task['type']); ?>">
                                    <?php echo h($task['type']); ?>
                                </span>
                            </td>
                            <td>
                                <?php echo $task['child_name'] ?? 'All Children'; ?>
                            </td>
                            <td>
                                <?php 
                                switch ($task['recurrence_type']) {
                                    case 'daily':
                                        echo 'Daily';
                                        break;
                                    case 'weekdays':
                                        echo 'Custom days';
                                        break;
                                    case 'weekly':
                                        echo 'Weekly';
                                        break;
                                    default:
                                        echo 'One-time';
                                }
                                ?>
                            </td>
                            <td>
                                <?php if ($task['is_active']): ?>
                                    <span class="badge badge-active">Active</span>
                                <?php else: ?>
                                    <span class="badge badge-archived">Inactive</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="actions">
                                    <a href="<?php echo APP_URL; ?>/admin/task_edit.php?id=<?php echo $task['id']; ?>" class="btn btn-sm btn-secondary">Edit</a>
                                    
                                    <form method="POST" style="display: inline;" onsubmit="return confirmDelete('Delete this task? This will also delete all task instances.')">
                                        <?php echo csrf_field(); ?>
                                        <input type="hidden" name="task_id" value="<?php echo $task['id']; ?>">
                                        <input type="hidden" name="action" value="delete">
                                        <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
