<?php
/**
 * Admin - Add/Edit Task
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

// Check if editing existing task
$task_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$task = null;
$is_edit = false;

if ($task_id > 0) {
    $task = get_task($db, $task_id);
    if (!$task) {
        set_flash('error', 'Task not found');
        redirect('admin/tasks.php');
    }
    $is_edit = true;
}

$page_title = $is_edit ? 'Edit Task' : 'Add Task';
$errors = [];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid request';
    } else {
        // Collect recurrence days if custom
        $recurrence_days = '';
        if (($_POST['recurrence_type'] ?? '') === 'weekdays' && !empty($_POST['recurrence_days'])) {
            $recurrence_days = implode(',', $_POST['recurrence_days']);
        }
        
        $data = [
            'title' => trim($_POST['title'] ?? ''),
            'emoji' => trim($_POST['emoji'] ?? ''),
            'description' => trim($_POST['description'] ?? ''),
            'type' => $_POST['type'] ?? 'chore',
            'assigned_child_id' => $_POST['assigned_child_id'] ?? '',
            'recurrence_type' => $_POST['recurrence_type'] ?? 'none',
            'recurrence_days' => $recurrence_days,
            'start_date' => $_POST['start_date'] ?? date('Y-m-d'),
            'end_date' => $_POST['end_date'] ?? '',
            'sort_order' => (int)($_POST['sort_order'] ?? 0),
            'is_active' => isset($_POST['is_active']) ? 1 : 0
        ];
        
        // Validate
        if (empty($data['title'])) {
            $errors[] = 'Title is required';
        }
        
        if (empty($data['start_date'])) {
            $errors[] = 'Start date is required';
        }
        
        // Save if no errors
        if (empty($errors)) {
            if ($is_edit) {
                update_task($db, $task_id, $data);
                set_flash('success', 'Task updated successfully');
            } else {
                create_task($db, $data);
                set_flash('success', 'Task added successfully');
            }
            redirect('admin/tasks.php');
        }
    }
}

// Get children for dropdown
$children = get_active_children($db);

// Get current values for form
$form_data = $task ?? [
    'title' => '',
    'emoji' => '',
    'description' => '',
    'type' => 'chore',
    'assigned_child_id' => '',
    'recurrence_type' => 'daily',
    'recurrence_days' => '',
    'start_date' => date('Y-m-d'),
    'end_date' => '',
    'sort_order' => 0,
    'is_active' => 1
];

// Parse recurrence days into array
$selected_days = [];
if (!empty($form_data['recurrence_days'])) {
    $selected_days = explode(',', $form_data['recurrence_days']);
}

// Override with POST data if there were errors
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $form_data['title'] = $_POST['title'] ?? '';
    $form_data['emoji'] = $_POST['emoji'] ?? '';
    $form_data['description'] = $_POST['description'] ?? '';
    $form_data['type'] = $_POST['type'] ?? 'chore';
    $form_data['assigned_child_id'] = $_POST['assigned_child_id'] ?? '';
    $form_data['recurrence_type'] = $_POST['recurrence_type'] ?? 'none';
    $form_data['start_date'] = $_POST['start_date'] ?? '';
    $form_data['end_date'] = $_POST['end_date'] ?? '';
    $form_data['sort_order'] = $_POST['sort_order'] ?? 0;
    $form_data['is_active'] = isset($_POST['is_active']) ? 1 : 0;
    $selected_days = $_POST['recurrence_days'] ?? [];
}

// Common emojis for quick selection
$common_emojis = ['📚', '✏️', '📖', '🔢', '🔬', '🎨', '🎵', '🏃', '🧹', '🍽️', '🛏️', '🐕', '🌱', '🗑️', '👕', '🦷', '📝', '🎮', '⭐', '✅'];

require_once __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1><?php echo $is_edit ? 'Edit Task' : 'Add Task'; ?></h1>
    <a href="<?php echo APP_URL; ?>/admin/tasks.php" class="btn btn-secondary">← Back to Tasks</a>
</div>

<div class="card" style="max-width: 700px;">
    <?php if (!empty($errors)): ?>
        <div class="alert alert-error">
            <ul style="margin: 0; padding-left: 20px;">
                <?php foreach ($errors as $error): ?>
                    <li><?php echo h($error); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>
    
    <form method="POST">
        <?php echo csrf_field(); ?>
        
        <!-- Title & Emoji -->
        <div class="form-row">
            <div class="form-group" style="flex: 0 0 100px;">
                <label for="emoji">Emoji</label>
                <input type="text" id="emoji" name="emoji" value="<?php echo h($form_data['emoji']); ?>" maxlength="10" style="font-size: 24px; text-align: center;">
            </div>
            <div class="form-group" style="flex: 1;">
                <label for="title">Title *</label>
                <input type="text" id="title" name="title" value="<?php echo h($form_data['title']); ?>" required placeholder="e.g., Math worksheet">
            </div>
        </div>
        
        <!-- Quick emoji picker -->
        <div class="form-group" style="margin-top: -10px;">
            <div style="display: flex; flex-wrap: wrap; gap: 8px;">
                <?php foreach ($common_emojis as $emoji): ?>
                    <button type="button" onclick="document.getElementById('emoji').value='<?php echo $emoji; ?>'" 
                            style="font-size: 20px; padding: 6px 10px; border: 1px solid #e2e8f0; border-radius: 6px; background: #fff; cursor: pointer;">
                        <?php echo $emoji; ?>
                    </button>
                <?php endforeach; ?>
            </div>
        </div>
        
        <div class="form-group">
            <label for="description">Description (optional)</label>
            <textarea id="description" name="description" rows="2" placeholder="Additional details about this task"><?php echo h($form_data['description']); ?></textarea>
        </div>
        
        <div class="form-row">
            <div class="form-group">
                <label for="type">Type *</label>
                <select id="type" name="type" required>
                    <option value="homeschool" <?php echo $form_data['type'] === 'homeschool' ? 'selected' : ''; ?>>📚 Homeschool</option>
                    <option value="chore" <?php echo $form_data['type'] === 'chore' ? 'selected' : ''; ?>>🧹 Chore</option>
                </select>
            </div>
            
            <div class="form-group">
                <label for="assigned_child_id">Assigned To</label>
                <select id="assigned_child_id" name="assigned_child_id">
                    <option value="">All Children</option>
                    <?php foreach ($children as $child): ?>
                        <option value="<?php echo $child['id']; ?>" <?php echo $form_data['assigned_child_id'] == $child['id'] ? 'selected' : ''; ?>>
                            <?php echo h($child['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
        
        <div class="form-group">
            <label for="recurrence_type">Recurrence *</label>
            <select id="recurrence_type" name="recurrence_type" required onchange="toggleDaysSelector()">
                <option value="none" <?php echo $form_data['recurrence_type'] === 'none' ? 'selected' : ''; ?>>One-time only</option>
                <option value="daily" <?php echo $form_data['recurrence_type'] === 'daily' ? 'selected' : ''; ?>>Daily</option>
                <option value="weekdays" <?php echo $form_data['recurrence_type'] === 'weekdays' ? 'selected' : ''; ?>>Specific days</option>
            </select>
        </div>
        
        <!-- Day selector (shown when recurrence_type is weekdays) -->
        <div class="form-group" id="days-selector" style="display: <?php echo $form_data['recurrence_type'] === 'weekdays' ? 'block' : 'none'; ?>;">
            <label>Select Days</label>
            <div style="display: flex; gap: 8px; flex-wrap: wrap;">
                <?php 
                $days = ['0' => 'Sun', '1' => 'Mon', '2' => 'Tue', '3' => 'Wed', '4' => 'Thu', '5' => 'Fri', '6' => 'Sat'];
                foreach ($days as $value => $label): 
                ?>
                    <label style="display: flex; align-items: center; gap: 6px; padding: 8px 12px; border: 2px solid #e2e8f0; border-radius: 8px; cursor: pointer;">
                        <input type="checkbox" name="recurrence_days[]" value="<?php echo $value; ?>" 
                               <?php echo in_array($value, $selected_days) ? 'checked' : ''; ?>>
                        <?php echo $label; ?>
                    </label>
                <?php endforeach; ?>
            </div>
        </div>
        
        <div class="form-row">
            <div class="form-group">
                <label for="start_date">Start Date *</label>
                <input type="date" id="start_date" name="start_date" value="<?php echo h($form_data['start_date']); ?>" required>
            </div>
            
            <div class="form-group">
                <label for="end_date">End Date (optional)</label>
                <input type="date" id="end_date" name="end_date" value="<?php echo h($form_data['end_date']); ?>">
                <small>Leave blank for no end date</small>
            </div>
        </div>
        
        <div class="form-row">
            <div class="form-group">
                <label for="sort_order">Sort Order</label>
                <input type="number" id="sort_order" name="sort_order" value="<?php echo (int)$form_data['sort_order']; ?>" min="0">
                <small>Lower numbers appear first</small>
            </div>
            
            <div class="form-group">
                <label>&nbsp;</label>
                <label class="form-check" style="padding-top: 8px;">
                    <input type="checkbox" name="is_active" value="1" <?php echo $form_data['is_active'] ? 'checked' : ''; ?>>
                    <span>Active</span>
                </label>
                <small>Inactive tasks won't generate new instances</small>
            </div>
        </div>
        
        <div style="display: flex; gap: 12px; margin-top: 24px;">
            <button type="submit" class="btn btn-primary">
                <?php echo $is_edit ? 'Update Task' : 'Add Task'; ?>
            </button>
            <a href="<?php echo APP_URL; ?>/admin/tasks.php" class="btn btn-secondary">Cancel</a>
        </div>
    </form>
</div>

<script>
function toggleDaysSelector() {
    const recurrenceType = document.getElementById('recurrence_type').value;
    const daysSelector = document.getElementById('days-selector');
    daysSelector.style.display = recurrenceType === 'weekdays' ? 'block' : 'none';
}
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
