<?php
/**
 * Admin Dashboard
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

$page_title = 'Dashboard';

// Get today's date
$today = date('Y-m-d');

// Get all active children
$children = get_active_children($db);

// Get stats for each child
$child_stats = [];
foreach ($children as $child) {
    $stats = get_child_task_stats($db, $child['id'], $today);
    $child_stats[$child['id']] = $stats;
}

// Count totals
$total_tasks = 0;
$total_completed = 0;
foreach ($child_stats as $stats) {
    $total_tasks += $stats['total'];
    $total_completed += $stats['completed'];
}

require_once __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1>Dashboard</h1>
    <span style="color: #64748b;"><?php echo date('l, F j, Y'); ?></span>
</div>

<!-- Overview Cards -->
<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 24px;">
    <div class="card" style="margin-bottom: 0;">
        <div style="color: #64748b; font-size: 14px; margin-bottom: 4px;">Total Tasks Today</div>
        <div style="font-size: 36px; font-weight: 700; color: #1e293b;"><?php echo $total_tasks; ?></div>
    </div>
    
    <div class="card" style="margin-bottom: 0;">
        <div style="color: #64748b; font-size: 14px; margin-bottom: 4px;">Completed</div>
        <div style="font-size: 36px; font-weight: 700; color: #22c55e;"><?php echo $total_completed; ?></div>
    </div>
    
    <div class="card" style="margin-bottom: 0;">
        <div style="color: #64748b; font-size: 14px; margin-bottom: 4px;">Remaining</div>
        <div style="font-size: 36px; font-weight: 700; color: #f59e0b;"><?php echo $total_tasks - $total_completed; ?></div>
    </div>
    
    <div class="card" style="margin-bottom: 0;">
        <div style="color: #64748b; font-size: 14px; margin-bottom: 4px;">Active Children</div>
        <div style="font-size: 36px; font-weight: 700; color: #6366f1;"><?php echo count($children); ?></div>
    </div>
</div>

<!-- Per-Child Progress -->
<div class="card">
    <div class="card-header">
        <h2>Today's Progress by Child</h2>
    </div>
    
    <?php if (empty($children)): ?>
        <div class="empty-state">
            <svg viewBox="0 0 24 24" fill="none" stroke-width="2" stroke-linecap="round">
                <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                <circle cx="9" cy="7" r="4"/>
                <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
            </svg>
            <h3>No Children Yet</h3>
            <p>Add children to start assigning tasks</p>
            <a href="<?php echo APP_URL; ?>/admin/child_edit.php" class="btn btn-primary" style="margin-top: 16px;">Add Child</a>
        </div>
    <?php else: ?>
        <div style="display: grid; gap: 16px;">
            <?php foreach ($children as $child): 
                $stats = $child_stats[$child['id']];
                $percentage = $stats['total'] > 0 ? round(($stats['completed'] / $stats['total']) * 100) : 0;
            ?>
                <div style="display: flex; align-items: center; gap: 16px; padding: 16px; background: #f8fafc; border-radius: 12px;">
                    <?php if (!empty($child['photo'])): ?>
                        <img src="<?php echo upload_url($child['photo']); ?>" alt="" class="avatar">
                    <?php else: ?>
                        <div class="avatar" style="background: <?php echo h($child['color']); ?>; display: flex; align-items: center; justify-content: center;">
                            <span style="color: #fff; font-weight: 600;"><?php echo strtoupper(substr($child['name'], 0, 1)); ?></span>
                        </div>
                    <?php endif; ?>
                    
                    <div style="flex: 1;">
                        <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                            <span style="font-weight: 600; color: #1e293b;"><?php echo h($child['name']); ?></span>
                            <span style="color: #64748b; font-size: 14px;">
                                <?php echo $stats['completed']; ?>/<?php echo $stats['total']; ?> tasks
                            </span>
                        </div>
                        <div style="height: 8px; background: #e2e8f0; border-radius: 4px; overflow: hidden;">
                            <div style="height: 100%; width: <?php echo $percentage; ?>%; background: <?php echo h($child['color']); ?>; border-radius: 4px;"></div>
                        </div>
                    </div>
                    
                    <div style="font-weight: 700; color: <?php echo $percentage == 100 ? '#22c55e' : '#64748b'; ?>; font-size: 18px;">
                        <?php echo $percentage; ?>%
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<!-- Quick Actions -->
<div class="card">
    <div class="card-header">
        <h2>Quick Actions</h2>
    </div>
    
    <div style="display: flex; gap: 12px; flex-wrap: wrap;">
        <a href="<?php echo APP_URL; ?>/admin/child_edit.php" class="btn btn-secondary">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="12" y1="5" x2="12" y2="19"/>
                <line x1="5" y1="12" x2="19" y2="12"/>
            </svg>
            Add Child
        </a>
        
        <a href="<?php echo APP_URL; ?>/admin/task_edit.php" class="btn btn-secondary">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="12" y1="5" x2="12" y2="19"/>
                <line x1="5" y1="12" x2="19" y2="12"/>
            </svg>
            Add Task
        </a>
        
        <a href="<?php echo APP_URL; ?>/" class="btn btn-secondary" target="_blank">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"/>
                <polyline points="15 3 21 3 21 9"/>
                <line x1="10" y1="14" x2="21" y2="3"/>
            </svg>
            Open Kid Dashboard
        </a>
        
        <a href="<?php echo APP_URL; ?>/admin/logout.php" class="btn btn-danger">
            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"/>
                <polyline points="16 17 21 12 16 7"/>
                <line x1="21" y1="12" x2="9" y2="12"/>
            </svg>
            Logout
        </a>
    </div>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
