<?php
/**
 * Admin - Children List
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

$page_title = 'Children';

// Handle archive/restore actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        set_flash('error', 'Invalid request');
    } else {
        $child_id = (int)$_POST['child_id'];
        
        if ($_POST['action'] === 'archive') {
            archive_child($db, $child_id);
            set_flash('success', 'Child archived successfully');
        } elseif ($_POST['action'] === 'restore') {
            restore_child($db, $child_id);
            set_flash('success', 'Child restored successfully');
        }
    }
    redirect('admin/children.php' . (isset($_GET['show_archived']) ? '?show_archived=1' : ''));
}

// Get children
$show_archived = isset($_GET['show_archived']);
$children = get_all_children($db, $show_archived);

require_once __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1>Children</h1>
    <a href="<?php echo APP_URL; ?>/admin/child_edit.php" class="btn btn-primary">
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <line x1="12" y1="5" x2="12" y2="19"/>
            <line x1="5" y1="12" x2="19" y2="12"/>
        </svg>
        Add Child
    </a>
</div>

<div class="card">
    <div class="card-header">
        <h2>All Children</h2>
        <div>
            <?php if ($show_archived): ?>
                <a href="<?php echo APP_URL; ?>/admin/children.php" class="btn btn-sm btn-secondary">Hide Archived</a>
            <?php else: ?>
                <a href="<?php echo APP_URL; ?>/admin/children.php?show_archived=1" class="btn btn-sm btn-secondary">Show Archived</a>
            <?php endif; ?>
        </div>
    </div>
    
    <?php if (empty($children)): ?>
        <div class="empty-state">
            <svg viewBox="0 0 24 24" fill="none" stroke-width="2" stroke-linecap="round">
                <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                <circle cx="9" cy="7" r="4"/>
            </svg>
            <h3>No Children Yet</h3>
            <p>Add your first child to get started</p>
        </div>
    <?php else: ?>
        <div class="table-wrapper">
            <table>
                <thead>
                    <tr>
                        <th>Photo</th>
                        <th>Name</th>
                        <th>Color</th>
                        <th>PIN</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($children as $child): ?>
                        <tr>
                            <td>
                                <?php if (!empty($child['photo'])): ?>
                                    <img src="<?php echo upload_url($child['photo']); ?>" alt="" class="avatar">
                                <?php else: ?>
                                    <div class="avatar" style="background: <?php echo h($child['color']); ?>; display: flex; align-items: center; justify-content: center;">
                                        <span style="color: #fff; font-weight: 600;"><?php echo strtoupper(substr($child['name'], 0, 1)); ?></span>
                                    </div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <strong><?php echo h($child['name']); ?></strong>
                            </td>
                            <td>
                                <span class="color-dot" style="background: <?php echo h($child['color']); ?>;"></span>
                            </td>
                            <td>
                                <?php echo $child['pin'] ? '••••' : '—'; ?>
                            </td>
                            <td>
                                <?php if ($child['is_archived']): ?>
                                    <span class="badge badge-archived">Archived</span>
                                <?php else: ?>
                                    <span class="badge badge-active">Active</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <div class="actions">
                                    <a href="<?php echo APP_URL; ?>/admin/child_edit.php?id=<?php echo $child['id']; ?>" class="btn btn-sm btn-secondary">Edit</a>
                                    
                                    <?php if ($child['is_archived']): ?>
                                        <form method="POST" style="display: inline;">
                                            <?php echo csrf_field(); ?>
                                            <input type="hidden" name="child_id" value="<?php echo $child['id']; ?>">
                                            <input type="hidden" name="action" value="restore">
                                            <button type="submit" class="btn btn-sm btn-primary">Restore</button>
                                        </form>
                                    <?php else: ?>
                                        <form method="POST" style="display: inline;" onsubmit="return confirmDelete('Archive this child? They will be hidden but not deleted.')">
                                            <?php echo csrf_field(); ?>
                                            <input type="hidden" name="child_id" value="<?php echo $child['id']; ?>">
                                            <input type="hidden" name="action" value="archive">
                                            <button type="submit" class="btn btn-sm btn-danger">Archive</button>
                                        </form>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
