<?php
/**
 * Admin - Add/Edit Child
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../functions.php';

// Check if editing existing child
$child_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$child = null;
$is_edit = false;

if ($child_id > 0) {
    $child = get_child($db, $child_id);
    if (!$child) {
        set_flash('error', 'Child not found');
        redirect('admin/children.php');
    }
    $is_edit = true;
}

$page_title = $is_edit ? 'Edit Child' : 'Add Child';
$errors = [];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Invalid request';
    } else {
        $data = [
            'name' => trim($_POST['name'] ?? ''),
            'color' => $_POST['color'] ?? '#6366f1',
            'pin' => trim($_POST['pin'] ?? ''),
            'sort_order' => (int)($_POST['sort_order'] ?? 0)
        ];
        
        // Validate
        if (empty($data['name'])) {
            $errors[] = 'Name is required';
        }
        
        if (!empty($data['pin']) && !preg_match('/^\d{4}$/', $data['pin'])) {
            $errors[] = 'PIN must be exactly 4 digits';
        }
        
        // Handle photo upload
        if (!empty($_FILES['photo']['name'])) {
            $upload_result = handle_photo_upload($_FILES['photo'], 'child');
            if ($upload_result['success']) {
                // Delete old photo if editing
                if ($is_edit && !empty($child['photo'])) {
                    delete_uploaded_file($child['photo']);
                }
                $data['photo'] = $upload_result['filename'];
            } else {
                $errors[] = 'Photo upload failed: ' . $upload_result['error'];
            }
        }
        
        // Save if no errors
        if (empty($errors)) {
            if ($is_edit) {
                update_child($db, $child_id, $data);
                set_flash('success', 'Child updated successfully');
            } else {
                create_child($db, $data);
                set_flash('success', 'Child added successfully');
            }
            redirect('admin/children.php');
        }
    }
}

// Get current values for form
$form_data = $child ?? [
    'name' => '',
    'photo' => '',
    'color' => '#6366f1',
    'pin' => '',
    'sort_order' => 0
];

// Override with POST data if there were errors
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $form_data['name'] = $_POST['name'] ?? '';
    $form_data['color'] = $_POST['color'] ?? '#6366f1';
    $form_data['pin'] = $_POST['pin'] ?? '';
    $form_data['sort_order'] = $_POST['sort_order'] ?? 0;
}

require_once __DIR__ . '/includes/header.php';
?>

<div class="page-header">
    <h1><?php echo $is_edit ? 'Edit Child' : 'Add Child'; ?></h1>
    <a href="<?php echo APP_URL; ?>/admin/children.php" class="btn btn-secondary">← Back to Children</a>
</div>

<div class="card" style="max-width: 600px;">
    <?php if (!empty($errors)): ?>
        <div class="alert alert-error">
            <ul style="margin: 0; padding-left: 20px;">
                <?php foreach ($errors as $error): ?>
                    <li><?php echo h($error); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>
    
    <form method="POST" enctype="multipart/form-data">
        <?php echo csrf_field(); ?>
        
        <div class="form-group">
            <label for="name">Name *</label>
            <input type="text" id="name" name="name" value="<?php echo h($form_data['name']); ?>" required>
        </div>
        
        <div class="form-group">
            <label for="photo">Photo</label>
            <?php if (!empty($form_data['photo'])): ?>
                <div style="margin-bottom: 12px;">
                    <img src="<?php echo upload_url($form_data['photo']); ?>" alt="" class="avatar avatar-lg">
                    <span style="margin-left: 12px; color: #64748b;">Current photo</span>
                </div>
            <?php endif; ?>
            <input type="file" id="photo" name="photo" accept="image/*">
            <small>Upload a photo of the child (JPG, PNG, GIF, WebP - max 5MB)</small>
        </div>
        
        <div class="form-row">
            <div class="form-group">
                <label for="color">Color</label>
                <input type="color" id="color" name="color" value="<?php echo h($form_data['color']); ?>" style="height: 44px; padding: 4px;">
                <small>Used for accents and progress bars</small>
            </div>
            
            <div class="form-group">
                <label for="pin">PIN (optional)</label>
                <input type="text" id="pin" name="pin" value="<?php echo h($form_data['pin']); ?>" maxlength="4" pattern="\d{4}" inputmode="numeric">
                <small>4-digit PIN to access their tasks</small>
            </div>
        </div>
        
        <div class="form-group">
            <label for="sort_order">Sort Order</label>
            <input type="number" id="sort_order" name="sort_order" value="<?php echo (int)$form_data['sort_order']; ?>" min="0">
            <small>Lower numbers appear first in navigation</small>
        </div>
        
        <div style="display: flex; gap: 12px; margin-top: 24px;">
            <button type="submit" class="btn btn-primary">
                <?php echo $is_edit ? 'Update Child' : 'Add Child'; ?>
            </button>
            <a href="<?php echo APP_URL; ?>/admin/children.php" class="btn btn-secondary">Cancel</a>
        </div>
    </form>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
